%Fluorescencewrapper is a script that guides the user through processing
%and analyzing a group of fluorescence traces obtained by GPatch, using
%datreader
%Read in the file
h3=msgbox('Please choose a fluorescence file.');
rawdata=datreader('d','a','g');
name='d';
delete(h3);

%Select a group
prompt= 'What is the fluorescence group number?  ';
group=input(prompt);
groupinfo=rawdata.(['group' num2str(group)]);

%Segment out parts of name for variable labelling
filenameglobal=strsplit(filenameglobal, '.');
filenameglobal=filenameglobal{1,1};

%Now format the name so it works as a matlab variable (no blank spaces, must start with
%letter) (In retrospect, I could have tried genvarname, but this seems to
%work quite well
filenameglobal=filenameglobal(filenameglobal~=' '); %Remove spaces
expression='[0123456789]'; %Set variable of all numerals
startIndex=regexp(filenameglobal,expression);
%Matlab variables must start with a letter, so if there is a number at the
%start of the filename, put an 'a' at the beginning of it
if startIndex(1,1)==1
    s1='a';
    filenameglobal=strcat(s1,filenameglobal,'_',num2str(group));
end


%Initialize data variables
num_sweeps=size(groupinfo,2); %Find number of sweeps in the group
num_points=size(groupinfo(1).data,1); %Find number of points in the sweep
groupdata=zeros([num_points num_sweeps]); %Preallocate group data variable for speed
groupvoltage=zeros([1 num_sweeps]); %Preallocate voltage data variable for speed

%This gets all the traces and puts them into a single array
for i=1:num_sweeps
    groupdata(:,i)=groupinfo(i).data;
    groupvoltage(:,i)=groupinfo(i).amp(2);
end

%Initialize pulse times
dur1=groupinfo(1).dur(1); %Duration of first pulse
dur2=groupinfo(1).dur(2); %duration of second pulse
dur3=groupinfo(1).dur(3); %Duration of third pulse
trig1pulse=groupinfo.trig1pulse; %Pulse that trigger is based off of
trig1delay=groupinfo.delay; %Delay of pulse
acq_period=groupinfo.acqrate; %Acquision rate (us per point)

%T0 is time zero, where the acquisition window starts. T1 is the end of the
%first time section (trigger delay and duration of the first pulse), and is
%equivalent to the beginning of the second pulse, which is the typical
%variable pulse in GPatch. T2 is the end of the second pulse/ beginning of the third pulse, and T3 is the
%end of the third pulse.
if trig1pulse==1
    T1=dur1+trig1delay;
    T2=dur1+dur2+trig1delay;
    T3=dur1+dur2+dur3+trig1delay;
elseif trig1pulse==2
    prompt= 'What is the trigger delay (in ms)?   ';
    trig1delay=input(prompt)*1000;
    T1=trig1delay;
    T2=dur2+trig1delay;
    T3=dur2+dur3+trig1delay;
end

%Determine if user wishes to filter additionally
filter_choice=menu('Do you want to apply a filter? WARNING: This will slightly shift the time course of your data.', 'Yes', 'No');
if filter_choice==1
    %Read in desired cutoff frequency from user
    cutoff_frequency_prompt= {'What is your desired cutoff frequency, in Hz?' 'Beta'};
    dlg_cutofftitle='Cutoff Frequency (Hz)';
    num_lines=1;
    %Default filter in 10 kHz
    default_cutoff={num2str(10000),'Beta'};
    cutoff_answer=inputdlg(cutoff_frequency_prompt, dlg_cutofftitle,num_lines,default_cutoff);
    cutoff_frequency=str2double(cutoff_answer{1});
    %Run bessel_filter_beta script to filter the data
    [filtered_data]=bessel_filter(groupdata,acq_period,cutoff_frequency);
    %Plot the original data and the filtered data for the user to compare
    figure
    plot(filtered_data, 'r')
    hold on
    plot(groupdata, 'b')
    %Ask the user if they want to accept the filtered data, if they say no,
    %end it... this is an opportunity to improve it, so the user does not
    %have to go back to the beginning
    accept_filter=menu('Do you want to accept the filtered data?', 'Yes', 'No');
    if accept_filter==1
        groupdata=filtered_data;
    end
end



%Allow user to choose for linear baseline or exponential baseline
baseline_type=menu('Choose a baseline type','Linear baseline', 'Exponential baseline');
if baseline_type==1

    %Fluo_baseline_start and _end provide values (which can be changed
    %by the user) that are read into fintegratelinear as the fitLimits value.
    default_fluo_baseline_start= (T1/1000)-6;
    default_fluo_baseline_end= (T1/1000)-1.5;
    %Fluo_examine_start and _end provide values that are read into fintegrate
    %as the intLimits value
    default_fluo_examine_start=T1/1000;
    default_fluo_examine_end=T2/1000;
    %Fluo_dark_start and _end provide values on the fluorescence level during
    %the dark noise period (when the shutter is closed, or no illumination) and
    %read them into fintegrate as the FoLimits value
    default_fluo_dark_start=(num_points*acq_period/1000)-6;
    default_fluo_dark_end=(num_points*acq_period/1000)-2;

    %Now initialize values for the dialog box
    fluo_initialvalues_prompt = {'Fluorescence baseline start','Fluorescence baseline end','Fluorescence plot start', 'Fluorescence plot end', 'Dark noise start', 'Dark noise end'};
    dlg_fluotitle = 'Fluorescence Initial Parameters';
    num_lines=1;
    default_fluo = {num2str(default_fluo_baseline_start), num2str(default_fluo_baseline_end), num2str(default_fluo_examine_start), num2str(default_fluo_examine_end), num2str(default_fluo_dark_start), num2str(default_fluo_dark_end)};
    fluoanswer=inputdlg(fluo_initialvalues_prompt,dlg_fluotitle,num_lines,default_fluo);

    %Now take data from dialog box and write new variables, converting into
    %points for fintegratelinear
    fluo_baseline_start= round(str2double(fluoanswer{1})*1000/acq_period);
    fluo_baseline_end = round(str2double(fluoanswer{2})*1000/acq_period);
    fluo_examine_start = round(str2double(fluoanswer{3})*1000/acq_period);
    fluo_examine_end = round(str2double(fluoanswer{4})*1000/acq_period);
    fluo_dark_start=round(str2double(fluoanswer{5})*1000/acq_period);
    fluo_dark_end=round(str2double(fluoanswer{6})*1000/acq_period);

    %Use these variables to feed into fintegrate
    intLimits_values=[fluo_examine_start fluo_examine_end];
    fitLimits_values=[fluo_baseline_start fluo_baseline_end];
    FoLimits_values=[fluo_dark_start fluo_dark_end];

    %Run fintegratelinear
    [fintegrate_data, deltaFData, deltaF]=fintegratelinear(groupdata, intLimits_values,fitLimits_values, FoLimits_values);

else
    %Fluo_baseline_1_start, 1_end, 2_start, and 2_end provide values (which can be changed
    %by the user) that are read into fintegrateexp as the fitLimits value.
    %1 designates the first section of fluorescence used for the exp fit
    %baseline, 2 designates the second
    default_fluo_baseline_1_start= (T1/1000)-6;
    default_fluo_baseline_1_end= (T1/1000)-1.5;
    default_fluo_baseline_2_start=(T3/1000)-6;
    default_fluo_baseline_2_end=(T3/1000)-1.5;

    %Fluo_examine_start and _end provide values that are read into fintegrate
    %as the intLimits value
    default_fluo_examine_start=T1/1000;
    default_fluo_examine_end=T2/1000;
    %Fluo_dark_start and _end provide values on the fluorescence level during
    %the dark noise period (when the shutter is closed, or no illumination) and
    %read them into fintegrate as the FoLimits value
    default_fluo_dark_start=(num_points*acq_period/1000)-6;
    default_fluo_dark_end=(num_points*acq_period/1000)-2;

    %Now initialize values for the dialog box
    fluo_initialvalues_prompt = {'Fluorescence baseline 1 start','Fluorescence baseline 1 end', 'Fluorescence baseline 2 start', 'Fluorescence baseline 2 end', 'Fluorescence plot start', 'Fluorescence plot end', 'Dark noise start', 'Dark noise end'};
    dlg_fluotitle = 'Fluorescence Initial Parameters';
    num_lines=1;
    default_fluo = {num2str(default_fluo_baseline_1_start), num2str(default_fluo_baseline_1_end),num2str(default_fluo_baseline_2_start),num2str(default_fluo_baseline_2_end), num2str(default_fluo_examine_start), num2str(default_fluo_examine_end), num2str(default_fluo_dark_start), num2str(default_fluo_dark_end)};
    fluoanswer=inputdlg(fluo_initialvalues_prompt,dlg_fluotitle,num_lines,default_fluo);

    %Now take data from dialog box and write new variables, converting into
    %points for fintegrate
    fluo_baseline_1_start= round(str2double(fluoanswer{1})*1000/acq_period);
    fluo_baseline_1_end = round(str2double(fluoanswer{2})*1000/acq_period);
    fluo_baseline_2_start=round(str2double(fluoanswer{3})*1000/acq_period);
    fluo_baseline_2_end=round(str2double(fluoanswer{4})*1000/acq_period);
    fluo_examine_start = round(str2double(fluoanswer{5})*1000/acq_period);
    fluo_examine_end = round(str2double(fluoanswer{6})*1000/acq_period);
    fluo_dark_start=round(str2double(fluoanswer{7})*1000/acq_period);
    fluo_dark_end=round(str2double(fluoanswer{8})*1000/acq_period);

    %Use these variables to feed into fintegrate
    intLimits_values=[fluo_examine_start fluo_examine_end];
    fitLimits_values=[fluo_baseline_1_start fluo_baseline_1_end fluo_baseline_2_start fluo_baseline_2_end];
    FoLimits_values=[fluo_dark_start fluo_dark_end];

    %Run fintegrateexp
    [fintegrate_data, deltaFData, deltaF]=fintegrateexp(groupdata, intLimits_values,fitLimits_values, FoLimits_values);
end

%%%%%%%%%%%%Now back to code for both baseline versions%%%%%%%%%%%%%
%Initialize fvdelayfit variables (basically, the time array)
time=(groupinfo(1).time(intLimits_values(1):intLimits_values(2))-(intLimits_values(1)-1)*acq_period)/1000;

%Run fvdelayfits
fit_1exp=fvdelayfit(time,fintegrate_data);
fit_2exp=fvdelayfit2exp(time,fintegrate_data);

%Add data to structured array by file name
f_data.fvoltage.(filenameglobal)=groupvoltage';
f_data.fvs.(filenameglobal)=deltaF';
f_data.ftraces.(filenameglobal)=deltaFData;
f_data.ffits_1exp.(filenameglobal)=fit_1exp;
f_data.ffits_2exp.(filenameglobal)=fit_2exp;

%Now ask the user if they want to export data, and if yes, use
%data_exporter to do so
export_data=menu('Do you want to export data to excel?','Yes', 'No');
if export_data==1
    fluo_data_exporter(f_data);
end

